#!/bin/bash

# nginx-rtmp Setup Script for GoPro Streaming
# Run with: sudo bash setup-rtmp-server.sh

set -e  # Exit on any error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Logging function
log() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

header() {
    echo -e "${BLUE}================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}================================${NC}"
}

# Check if running as root
if [[ $EUID -eq 0 ]]; then
   error "This script should not be run as root. Run with: sudo bash setup-rtmp-server.sh"
   exit 1
fi

# Check if sudo is available
if ! command -v sudo &> /dev/null; then
    error "sudo is required but not installed"
    exit 1
fi

header "nginx-rtmp GoPro Streaming Server Setup"

# Get server IP
SERVER_IP=$(hostname -I | awk '{print $1}')
log "Detected server IP: $SERVER_IP"

header "1. Updating System"
sudo apt update && sudo apt upgrade -y
log "System updated successfully"

header "2. Installing Required Packages"
sudo apt install -y nginx libnginx-mod-rtmp ffmpeg wget curl ufw
log "Packages installed successfully"

header "3. Backing Up Original nginx Configuration"
sudo cp /etc/nginx/nginx.conf /etc/nginx/nginx.conf.backup.$(date +%Y%m%d_%H%M%S)
log "Original nginx.conf backed up"

header "4. Creating nginx-rtmp Configuration"
sudo tee /etc/nginx/nginx.conf > /dev/null <<EOF
user www-data;
worker_processes auto;
pid /run/nginx.pid;
include /etc/nginx/modules-enabled/*.conf;

events {
    worker_connections 1024;
    use epoll;
    multi_accept on;
}

# RTMP Configuration - Streaming Only
rtmp {
    server {
        listen 1935;
        chunk_size 4096;
        allow publish all;
        allow play all;
        
        application live {
            live on;
            
            # Optimize for low latency streaming
            sync 10ms;
            
            # NO RECORDING - pure streaming relay
            record off;
            
            # Optimize buffer for real-time streaming
            buffer 100ms;
            
            # Enable metadata passthrough
            meta copy;
            
            # Limit concurrent connections per stream
            max_connections 10;
        }
    }
}

# HTTP Configuration for monitoring
http {
    sendfile on;
    tcp_nopush on;
    tcp_nodelay on;
    keepalive_timeout 65;
    types_hash_max_size 2048;
    
    include /etc/nginx/mime.types;
    default_type application/octet-stream;
    
    # Minimal logging for performance
    access_log /var/log/nginx/access.log;
    error_log /var/log/nginx/error.log warn;
    
    # RTMP Statistics and monitoring
    server {
        listen 8080;
        
        # RTMP statistics page
        location /stat {
            rtmp_stat all;
            rtmp_stat_stylesheet stat.xsl;
            add_header Access-Control-Allow-Origin * always;
        }
        
        location /stat.xsl {
            root /usr/share/nginx/html;
        }
        
        # Health check
        location /health {
            return 200 "nginx-rtmp streaming server running\n";
            add_header Content-Type text/plain;
        }
    }
}
EOF
log "nginx-rtmp configuration created"

header "5. Downloading RTMP Statistics Stylesheet"
sudo mkdir -p /usr/share/nginx/html
sudo wget -q -O /usr/share/nginx/html/stat.xsl https://raw.githubusercontent.com/arut/nginx-rtmp-module/master/stat.xsl
log "RTMP statistics stylesheet downloaded"

header "6. Configuring Firewall"
# Reset UFW to default state
sudo ufw --force reset

# Set default policies
sudo ufw default deny incoming
sudo ufw default allow outgoing

# Allow SSH (critical!)
sudo ufw allow ssh comment 'SSH access'

# Allow RTMP port for GoPros
sudo ufw allow 1935/tcp comment 'RTMP for GoPros'

# Allow HTTP monitoring interface
sudo ufw allow 8080/tcp comment 'nginx-rtmp monitoring'

# Enable firewall
sudo ufw --force enable
log "Firewall configured and enabled"

header "7. Testing nginx Configuration"
if sudo nginx -t; then
    log "nginx configuration test passed"
else
    error "nginx configuration test failed!"
    exit 1
fi

header "8. Starting nginx Service"
sudo systemctl restart nginx
sudo systemctl enable nginx

# Wait a moment for service to start
sleep 2

if sudo systemctl is-active --quiet nginx; then
    log "nginx service started successfully"
else
    error "Failed to start nginx service"
    sudo systemctl status nginx
    exit 1
fi

header "9. Verifying RTMP Port"
if sudo netstat -tlnp | grep -q :1935; then
    log "RTMP port 1935 is listening"
else
    warn "RTMP port 1935 may not be listening properly"
fi

header "10. Performance Optimization"
# Increase file descriptor limits
echo "* soft nofile 65536" | sudo tee -a /etc/security/limits.conf
echo "* hard nofile 65536" | sudo tee -a /etc/security/limits.conf

# Optimize network settings for streaming
echo "net.core.rmem_max = 134217728" | sudo tee -a /etc/sysctl.conf
echo "net.core.wmem_max = 134217728" | sudo tee -a /etc/sysctl.conf
sudo sysctl -p > /dev/null 2>&1
log "Performance optimization applied"

header "Setup Complete!"

echo ""
echo -e "${GREEN}✅ nginx-rtmp server is now running!${NC}"
echo ""
echo -e "${BLUE}Server Information:${NC}"
echo "  Server IP: $SERVER_IP"
echo "  RTMP Port: 1935"
echo "  Monitoring: http://$SERVER_IP:8080"
echo ""
echo -e "${BLUE}GoPro Stream URLs:${NC}"
echo "  GoPro 1: rtmp://$SERVER_IP:1935/live/gopro1"
echo "  GoPro 2: rtmp://$SERVER_IP:1935/live/gopro2"
echo "  GoPro 3: rtmp://$SERVER_IP:1935/live/gopro3"
echo "  GoPro 4: rtmp://$SERVER_IP:1935/live/gopro4"
echo ""
echo -e "${BLUE}OBS Media Source URLs:${NC}"
echo "  Use the same URLs as above in OBS Media Sources"
echo ""
echo -e "${BLUE}Monitoring URLs:${NC}"
echo "  Stream Statistics: http://$SERVER_IP:8080/stat"
echo "  Health Check: http://$SERVER_IP:8080/health"
echo ""
echo -e "${BLUE}Useful Commands:${NC}"
echo "  Check nginx status: sudo systemctl status nginx"
echo "  View nginx logs: sudo tail -f /var/log/nginx/error.log"
echo "  Test stream: ffplay rtmp://$SERVER_IP:1935/live/gopro1"
echo "  Check firewall: sudo ufw status"
echo ""
echo -e "${YELLOW}Next Steps:${NC}"
echo "1. Configure your GoPros to stream to the URLs above"
echo "2. Add Media Sources in OBS using the same URLs"
echo "3. Monitor streams at http://$SERVER_IP:8080/stat"
echo ""
echo -e "${GREEN}Setup completed successfully!${NC}"